package com.clarkparsia.modularity.locality;

import org.semanticweb.owl.model.*;

import java.util.Collection;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.Set;


/**
 * <p/>
 * Title: </p>
 * <p/>
 * Description: Implements syntactic locality evaluation for axioms </p>
 * <p/>
 * Copyright: Copyright (c) 2007 </p>
 * <p/>
 * Company: Clark & Parsia, LLC. <http://www.clarkparsia.com> </p>
 *
 * @author Mike Smith
 * <p/>
 * Essential bug fixes by Thomas Schneider, School of Computer Science, University of Manchester 
 */
public class SyntacticLocalityEvaluator implements LocalityEvaluator {

    private LocalityClass localityCls;

    private AxiomLocalityVisitor axiomVisitor;

    private static EnumSet<LocalityClass> supportedLocalityClasses = EnumSet.of(LocalityClass.TOP_BOTTOM,
                                                                                LocalityClass.BOTTOM_BOTTOM,
                                                                                LocalityClass.TOP_TOP);


    public SyntacticLocalityEvaluator(LocalityClass localityClass) {
        this.localityCls = localityClass;
        this.axiomVisitor = new AxiomLocalityVisitor();
        if (!supportedLocalityClasses.contains(localityClass)) {
            throw new RuntimeException("Unsupported locality class: " + localityClass);
        }
    }


    public Set<LocalityClass> supportedLocalityClasses() {
        return supportedLocalityClasses;
    }


    private class AxiomLocalityVisitor implements OWLAxiomVisitor {

        private BottomEquivalenceEvaluator bottomEvaluator;

        private boolean isLocal;

        private Collection<? extends OWLEntity> signature;

        private TopEquivalenceEvaluator topEvaluator;


        public AxiomLocalityVisitor() {
            bottomEvaluator = new BottomEquivalenceEvaluator();
            topEvaluator = new TopEquivalenceEvaluator();
            topEvaluator.setBottomEvaluator(bottomEvaluator);
            bottomEvaluator.setTopEvaluator(topEvaluator);
        }


        public boolean isLocal(OWLAxiom axiom, Collection<? extends OWLEntity> signature) {
            this.signature = signature;
            isLocal = false;
            axiom.accept(this);
            return isLocal;
        }


        // BUGFIX: (TS) Antisymm OP axioms are local in the *_BOTTOM case:
        //              The empty object property is antisymmetric!
        public void visit(OWLAntiSymmetricObjectPropertyAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = true;
                    break;
                case TOP_TOP:
                    isLocal = false;
                    break;
            }
        }

        public void visit(OWLAxiomAnnotationAxiom axiom) {
            isLocal = true;
        }


        public void visit(OWLClassAssertionAxiom axiom) {
            isLocal = topEvaluator.isTopEquivalent(axiom.getDescription(), signature, localityCls);
        }


        public void visit(OWLDataPropertyAssertionAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = false;
                    break;
                case TOP_TOP:
                    isLocal = !signature.contains(axiom.getProperty().asOWLDataProperty());
                    break;
            }
        }


        public void visit(OWLDataPropertyDomainAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = !signature.contains(axiom.getProperty().asOWLDataProperty()) || topEvaluator.isTopEquivalent(
                            axiom.getDomain(),
                            signature,
                            localityCls);
                    break;
                case TOP_TOP:
                    isLocal = topEvaluator.isTopEquivalent(axiom.getDomain(), signature, localityCls);
                    break;
            }
        }


        public void visit(OWLDataPropertyRangeAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = !signature.contains(axiom.getProperty().asOWLDataProperty());
                    break;
                case TOP_TOP:
                    isLocal = false;
                    break;
            }
        }


        public void visit(OWLDataSubPropertyAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = !signature.contains(axiom.getSubProperty().asOWLDataProperty());
                    break;
                case TOP_TOP:
                    isLocal = !signature.contains(axiom.getSuperProperty().asOWLDataProperty());
                    break;
            }
        }


        // BUGFIX: (TS) Individual declaration axioms are local, too.
        //              They need to be added to the module after the locality checks have been performed. 
        public void visit(OWLDeclarationAxiom axiom) {
            isLocal = true;
//            isLocal = !(axiom.getEntity().isOWLIndividual());
        }


        // BUGFIX: (TS) Different individuals axioms are local, too.
        //              They need to be added to the module after the locality checks have been performed.
        public void visit(OWLDifferentIndividualsAxiom axiom) {
            isLocal = true;
//            isLocal = false;
        }


        // BUGFIX: (TS) An n-ary disj classes axiom is local
        //              iff at most one of the involved class descriptions is not bot-equivalent.
        public void visit(OWLDisjointClassesAxiom axiom) {
            Collection<OWLDescription> disjs = axiom.getDescriptions();
            int size = disjs.size();
            if (size == 1) {
                throw new RuntimeException("Unary disjoint axiom.");
            }
            else {
                boolean nonBottomEquivDescFound = false;
                for (OWLDescription desc : disjs) {
                    if (!bottomEvaluator.isBottomEquivalent(desc, signature, localityCls)) {
                        if (nonBottomEquivDescFound) {
                            isLocal = false;
                            return;
                        }
                        else {
                            nonBottomEquivDescFound = true;
                        }
                    }
                }
            }
            isLocal = true;
        }

        // BUGFIX (TS): Added the case where it *is* local
        public void visit(OWLDisjointDataPropertiesAxiom axiom) {
            switch (localityCls){
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    Collection<OWLDataPropertyExpression> disjs = axiom.getProperties();
                    int size = disjs.size();
                    if (size == 1) {
                        throw new RuntimeException("Unary disjoint axiom.");
                    }
                    else {
                        boolean nonBottomEquivPropFound = false;
                        for (OWLDataPropertyExpression dpe : disjs) {
                            if (signature.contains(dpe.asOWLDataProperty())) {
                                if (nonBottomEquivPropFound) {
                                    isLocal = false;
                                    return;
                                }
                                else {
                                    nonBottomEquivPropFound = true;
                                }
                            }
                        }
                    }
                    isLocal = true;
                    break;
                case TOP_TOP:
                    isLocal = false;
                    break;
            }
        }


        // BUGFIX (TS): Added the case where it *is* local
        public void visit(OWLDisjointObjectPropertiesAxiom axiom) {
            switch (localityCls){
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    Collection<OWLObjectPropertyExpression> disjs = axiom.getProperties();
                    int size = disjs.size();
                    if (size == 1) {
                        throw new RuntimeException("Unary disjoint axiom.");
                    }
                    else {
                        boolean nonBottomEquivPropFound = false;
                        for (OWLObjectPropertyExpression ope : disjs) {
                            if (signature.contains(ope.getNamedProperty())) {
                                if (nonBottomEquivPropFound) {
                                    isLocal = false;
                                    return;
                                }
                                else {
                                    nonBottomEquivPropFound = true;
                                }
                            }
                        }
                    }
                    isLocal = true;
                    break;
                case TOP_TOP:
                    isLocal = false;
                    break;
            }
        }


        // BUGFIX (TS): added the two cases where a disj union axiom *is* local:
        // - if LHS and all class descr on RHS are bot-equiv
        // - if RHS is top-equiv, one descr on LHS is top-equiv and the others are bot-equiv
        public void visit(OWLDisjointUnionAxiom axiom) {
            OWLClass lhs = axiom.getOWLClass();
            Collection<OWLDescription> rhs = axiom.getDescriptions();
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                    if (!signature.contains(lhs)) {
                        for (OWLDescription desc : rhs) {
                            if (!bottomEvaluator.isBottomEquivalent(desc, signature, localityCls)) {
                                isLocal = false;
                                return;
                            }
                        }
                        isLocal = true;
                    }
                    else {
                        isLocal = false;
                    }
                    break;
                case TOP_BOTTOM:
                case TOP_TOP:
                    if (!signature.contains(rhs)) {
                        boolean bottomEquivDescFound = false;
                        for (OWLDescription desc : rhs) {
                            if (!bottomEvaluator.isBottomEquivalent(desc, signature, localityCls)) {
                                if (topEvaluator.isTopEquivalent(desc, signature, localityCls)) {
                                    if (bottomEquivDescFound) {
                                        isLocal = false;
                                        return;
                                    }
                                    else {
                                        bottomEquivDescFound = true;
                                    }
                                }
                                else {
                                    isLocal = false;
                                    return;
                                }
                            }
                        }
                        isLocal = true;
                    }
                    else {
                        isLocal = false;
                    }
                    break;
            }
        }


        public void visit(OWLEntityAnnotationAxiom axiom) {
            isLocal = true;
        }


        public void visit(OWLEquivalentClassesAxiom axiom) {
            isLocal = true;

            Iterator<OWLDescription> eqs = axiom.getDescriptions().iterator();
            OWLDescription first = eqs.next();

            // axiom is local if it contains a single class description
            if (!eqs.hasNext())
                return;

            // axiom is local iff either all class descriptions evaluate to TOP
            // or all evaluate to BOTTOM

            // check if first class descr. is BOTTOM
            boolean isBottom = bottomEvaluator.isBottomEquivalent(first, signature, localityCls);

            // if not BOTTOM or not TOP then this axiom is non-local
            if (!isBottom && !topEvaluator.isTopEquivalent(first, signature, localityCls))
                isLocal = false;

//            // unless we find a non-locality, process all the class descriptions
//            while (isLocal && eqs.hasNext()) {
//                OWLDescription next = eqs.next();
//
//                if (isBottom) {
//                    // other concepts were BOTTOM so this one should be BOTTOM
//                    // too
//                    if (!bottomEvaluator.isBottomEquivalent(next, signature, localityCls)) {
//                        isLocal = false;
//                    }
//                }
//                else {
//                    // other concepts were TOP so this one should be TOP too
//                    if (!topEvaluator.isTopEquivalent(next, signature, localityCls)) {
//                        isLocal = false;
//                    }
//                }
//            }

            if (isBottom) {
                // unless we find a non-locality, process all the class descriptions
                while (isLocal && eqs.hasNext()) {
                    OWLDescription next = eqs.next();
                    // first class descr. was BOTTOM, so this one should be BOTTOM too
                    if (!bottomEvaluator.isBottomEquivalent(next, signature, localityCls)) {
                        isLocal = false;
                    }
                }
            }
            else {
                // unless we find a non-locality, process all the class descriptions
                while (isLocal && eqs.hasNext()) {
                    OWLDescription next = eqs.next();
                    // first class descr. was TOP, so this one should be TOP too
                    if (!topEvaluator.isTopEquivalent(next, signature, localityCls)) {
                        isLocal = false;
                    }
                }
            }
        }


        public void visit(OWLEquivalentDataPropertiesAxiom axiom) {
            Collection<OWLDataPropertyExpression> eqs = axiom.getProperties();
            int size = eqs.size();
            if (size == 1) {
                isLocal = true;
            }
            else {
                for (OWLDataPropertyExpression p : eqs) {
                    if (signature.contains(p.asOWLDataProperty())) {
                        isLocal = false;
                        return;
                    }
                }
                isLocal = true;
            }
        }


        public void visit(OWLEquivalentObjectPropertiesAxiom axiom) {
            Collection<OWLObjectPropertyExpression> eqs = axiom.getProperties();
            int size = eqs.size();
            if (size == 1) {
                isLocal = true;
            }
            else {
                for (OWLObjectPropertyExpression p : eqs) {
                    if (signature.contains(p.getNamedProperty())) {
                        isLocal = false;
                        return;
                    }
                }
                isLocal = true;
            }
        }


        public void visit(OWLFunctionalDataPropertyAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = !signature.contains(axiom.getProperty().asOWLDataProperty());
                    break;
                case TOP_TOP:
                    isLocal = false;
                    break;
            }
        }


        //BUGFIX (TS): replaced call to asOWLObjectProperty() by getNamedProperty()
        public void visit(OWLFunctionalObjectPropertyAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = !signature.contains(axiom.getProperty().getNamedProperty());
                    break;
                case TOP_TOP:
                    isLocal = false;
                    break;
            }
        }


        public void visit(OWLImportsDeclaration axiom) {
            isLocal = false;
        }


        public void visit(OWLInverseFunctionalObjectPropertyAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = !signature.contains(axiom.getProperty().getNamedProperty());
                    break;
                case TOP_TOP:
                    isLocal = false;
                    break;
            }
        }


        public void visit(OWLInverseObjectPropertiesAxiom axiom) {
            isLocal = !signature.contains(axiom.getFirstProperty().getNamedProperty()) && !signature.contains(axiom.getSecondProperty().getNamedProperty());
        }


        // BUGFIX: (TS) Irreflexive OP axioms are local in the *_BOTTOM case:
        //              The empty object property is irreflexive!
        public void visit(OWLIrreflexiveObjectPropertyAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = true;
                    break;
                case TOP_TOP:
                    isLocal = false;
                    break;
            }
        }


        // BUGFIX: (TS) Added the case where this is local. (This is dual to the case of a "positive" DP assertion.)
        public void visit(OWLNegativeDataPropertyAssertionAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = !signature.contains(axiom.getProperty().asOWLDataProperty());
                    break;
                case TOP_TOP:
                    isLocal = false;
                    break;
            }
        }


        // BUGFIX: (TS) Added the case where this is local. (This is dual to the case of a "positive" OP assertion.)
        public void visit(OWLNegativeObjectPropertyAssertionAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = !signature.contains(axiom.getProperty().getNamedProperty());
                    break;
                case TOP_TOP:
                    isLocal = false;
                    break;
            }
        }


        public void visit(OWLObjectPropertyAssertionAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = false;
                    break;
                case TOP_TOP:
                    isLocal = !signature.contains(axiom.getProperty().getNamedProperty());
                    break;
            }
        }


        // BUGFIX: (TS) Added the cases where this is local
        public void visit(OWLObjectPropertyChainSubPropertyAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    // Axiom is local iff at least one prop in the chain is bot-equiv
                    for (OWLObjectPropertyExpression ope : axiom.getPropertyChain()) {
                        if (!signature.contains(ope.getNamedProperty())) {
                            isLocal = true;
                            return;
                        }
                    }
                    isLocal = false;
                    break;
                case TOP_TOP:
                    // Axiom is local iff RHS is top-equiv
                    if (!signature.contains(axiom.getSuperProperty().getNamedProperty())) {
                        isLocal = true;
                    }
                    else {
                        isLocal = false;
                    }
                    break;
            }
        }


        public void visit(OWLObjectPropertyDomainAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = !signature.contains(axiom.getProperty().getNamedProperty()) || topEvaluator.isTopEquivalent(
                            axiom.getDomain(),
                            signature,
                            localityCls);
                    break;
                case TOP_TOP:
                    isLocal = topEvaluator.isTopEquivalent(axiom.getDomain(), signature, localityCls);
                    break;
            }
        }


        public void visit(OWLObjectPropertyRangeAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = !signature.contains(axiom.getProperty().getNamedProperty()) || topEvaluator.isTopEquivalent(
                            axiom.getRange(),
                            signature,
                            localityCls);
                    break;
                case TOP_TOP:
                    isLocal = topEvaluator.isTopEquivalent(axiom.getRange(), signature, localityCls);
                    break;
            }
        }


        public void visit(OWLObjectSubPropertyAxiom axiom) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isLocal = !signature.contains(axiom.getSubProperty().getNamedProperty());
                    break;
                case TOP_TOP:
                    isLocal = !signature.contains(axiom.getSuperProperty().getNamedProperty());
                    break;
            }
        }


        public void visit(OWLReflexiveObjectPropertyAxiom axiom) {
            isLocal = !signature.contains(axiom.getProperty().getNamedProperty());
        }


        public void visit(OWLOntologyAnnotationAxiom axiom) {
            isLocal = true;
        }


        // BUGFIX: (TS) Same individuals axioms are local, too.
        //              They need to be added to the module after the locality checks have been performed.
        public void visit(OWLSameIndividualsAxiom axiom) {
//            isLocal = false;
            isLocal = true;
        }


        public void visit(OWLSubClassAxiom axiom) {
            isLocal = bottomEvaluator.isBottomEquivalent(axiom.getSubClass(),
                                                         signature,
                                                         localityCls) || topEvaluator.isTopEquivalent(axiom.getSuperClass(),
                                                                                                      signature,
                                                                                                      localityCls);
        }


        public void visit(OWLSymmetricObjectPropertyAxiom axiom) {
            isLocal = !signature.contains(axiom.getProperty().getNamedProperty());
        }


        public void visit(OWLTransitiveObjectPropertyAxiom axiom) {
            isLocal = !signature.contains(axiom.getProperty().getNamedProperty());
        }


        //TODO: (TS) Can't we treat this in a more differentiated way?
        public void visit(SWRLRule axiom) {
            isLocal = false;
        }
    }


    /**
     * Used to determine if descriptions are equivalent to \bottom using the provided locality class
     */
    private static class BottomEquivalenceEvaluator implements OWLDescriptionVisitor {

        private boolean isBottomEquivalent;

        private LocalityClass localityCls;

        private Collection<? extends OWLEntity> signature;

        private TopEquivalenceEvaluator topEvaluator;


        public BottomEquivalenceEvaluator() {
        }


        private boolean isBottomEquivalent(OWLDescription desc) {
            desc.accept(this);
            return isBottomEquivalent;
        }


        public boolean isBottomEquivalent(OWLDescription desc, Collection<? extends OWLEntity> signature,
                                          LocalityClass localityCls) {
            this.localityCls = localityCls;
            this.signature = signature;
            desc.accept(this);
            return isBottomEquivalent;
        }


        public void setTopEvaluator(TopEquivalenceEvaluator evaluator) {
            this.topEvaluator = evaluator;
        }


        public void visit(OWLClass desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                    isBottomEquivalent = desc.isOWLNothing() || (!desc.isOWLThing() && !signature.contains(desc));
                    break;
                case TOP_BOTTOM:
                case TOP_TOP:
                    isBottomEquivalent = desc.isOWLNothing();
                    break;
            }
        }


        // BUGFIX: (TS) Even in the TOP_TOP case, this is not bottom-equiv:
        //              "forall top.D" is not necessarily empty   
        public void visit(OWLDataAllRestriction desc) {
//            switch (localityCls) {
//                case BOTTOM_BOTTOM:
//                case TOP_BOTTOM:
//                    isBottomEquivalent = false;
//                    break;
//                case TOP_TOP:
//                    // FIXME: This ignores TOP_DATA case
//                    isBottomEquivalent = !signature.contains(desc.getProperty().asOWLDataProperty());
//                    break;
//            }
            isBottomEquivalent = false;
        }


        // BUGFIX: (TS) Corrected both conditions; included case n==0
        public void visit(OWLDataExactCardinalityRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isBottomEquivalent = (desc.getCardinality() > 0)
                            && (!signature.contains(desc.getProperty().asOWLDataProperty()));
                    break;
                case TOP_TOP:
                    isBottomEquivalent = false;
                    break;
            }
        }


        // BUGFIX: (TS) A data max card restriction is never bottom-equiv.
        // TODO: (TS) If the filler can never be empty, then the TOP_TOP case can be bottom-equiv.
        public void visit(OWLDataMaxCardinalityRestriction desc) {
            isBottomEquivalent = false;
        }


        // BUGFIX: (TS) The *_BOTTOM case only works if n > 0.
        public void visit(OWLDataMinCardinalityRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isBottomEquivalent = (desc.getCardinality() > 0)
                        && (!signature.contains(desc.getProperty().asOWLDataProperty()));
                    break;
                case TOP_TOP:
                    isBottomEquivalent = false;
                    break;
            }
        }


        public void visit(OWLDataSomeRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isBottomEquivalent = !signature.contains(desc.getProperty().asOWLDataProperty());
                    break;
                case TOP_TOP:
                    isBottomEquivalent = false;
                    break;
            }
        }


        public void visit(OWLDataValueRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isBottomEquivalent = !signature.contains(desc.getProperty().asOWLDataProperty());
                    break;
                case TOP_TOP:
                    isBottomEquivalent = false;
                    break;
            }
        }


        // BUGFIX (TS): TOP_TOP case was missing the first conjunct
        public void visit(OWLObjectAllRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isBottomEquivalent = false;
                    break;
                case TOP_TOP:
                    isBottomEquivalent = !signature.contains(desc.getProperty().getNamedProperty()) && isBottomEquivalent(desc.getFiller());
                    break;
            }
        }


        public void visit(OWLObjectComplementOf desc) {
            isBottomEquivalent = topEvaluator.isTopEquivalent(desc.getOperand(), signature, localityCls);
        }


        // BUGFIX: (TS) Since an exact card restriction is a conjunction of a min and a max card restriction,
        //              there are cases where it is bottom-local
        public void visit(OWLObjectExactCardinalityRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isBottomEquivalent = (desc.getCardinality() > 0)
                            && (!signature.contains(desc.getProperty().getNamedProperty()))
                            || isBottomEquivalent(desc.getFiller());
                    break;
                case TOP_TOP:
                    isBottomEquivalent = (
                            (desc.getCardinality() > 0) && (
                                    isBottomEquivalent(desc.getFiller()) || (
                                            (!signature.contains(desc.getProperty().getNamedProperty()))
                                            && topEvaluator.isTopEquivalent(desc.getFiller(), signature, localityCls)
                                    )
                            )
                    );
                    break;
            }
        }


        public void visit(OWLObjectIntersectionOf desc) {
            for (OWLDescription conj : desc.getOperands()) {
                if (isBottomEquivalent(conj)) {
                    isBottomEquivalent = true;
                    return;
                }
            }
            isBottomEquivalent = false;
        }


        // BUGFIX (TS): Corrected all conditions.
        //              The n==0 case doesn't affect bottom-equivalence of this type of restriction,
        //              but n>0 does!
        public void visit(OWLObjectMaxCardinalityRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isBottomEquivalent = false;
                    break;
                case TOP_TOP:
                    isBottomEquivalent = (desc.getCardinality() > 0) 
                            && (!signature.contains(desc.getProperty().getNamedProperty()))
                            && topEvaluator.isTopEquivalent(desc.getFiller(), signature, localityCls);
                    break;
            }
        }


        // BUGFIX (TS): Corrected all conditions, considering the case n==0
        public void visit(OWLObjectMinCardinalityRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isBottomEquivalent = (desc.getCardinality() > 0)
                            && !signature.contains(desc.getProperty().getNamedProperty())
                            || isBottomEquivalent(desc.getFiller());
                    break;
                case TOP_TOP:
                    isBottomEquivalent = (desc.getCardinality() > 0)
                            && isBottomEquivalent(desc.getFiller());
                    break;
            }
        }


        public void visit(OWLObjectOneOf desc) {
            isBottomEquivalent = desc.getIndividuals().isEmpty();
        }


        public void visit(OWLObjectSelfRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isBottomEquivalent = !signature.contains(desc.getProperty().getNamedProperty());
                    break;
                case TOP_TOP:
                    isBottomEquivalent = false;
                    break;
            }
        }


        public void visit(OWLObjectSomeRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isBottomEquivalent = !signature.contains(desc.getProperty().getNamedProperty()) || isBottomEquivalent(
                            desc.getFiller());
                    break;
                case TOP_TOP:
                    isBottomEquivalent = isBottomEquivalent(desc.getFiller());
                    break;
            }
        }


        public void visit(OWLObjectUnionOf desc) {
            for (OWLDescription disj : desc.getOperands()) {
                if (!isBottomEquivalent(disj)) {
                    isBottomEquivalent = false;
                    return;
                }
            }
            isBottomEquivalent = true;
        }


        public void visit(OWLObjectValueRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isBottomEquivalent = !signature.contains(desc.getProperty().getNamedProperty()) || !signature.contains(
                            desc.getValue());
                    break;
                case TOP_TOP:
                    isBottomEquivalent = false;
                    break;
            }
        }
    }


    /**
     * Used to determine if descriptions are equivalent to \top using the provided locality class
     */
    private static class TopEquivalenceEvaluator implements OWLDescriptionVisitor {

        private BottomEquivalenceEvaluator bottomEvaluator;

        private boolean isTopEquivalent;

        private LocalityClass localityCls;

        private Collection<? extends OWLEntity> signature;


        public TopEquivalenceEvaluator() {
        }


        private boolean isTopEquivalent(OWLDescription desc) {
            desc.accept(this);
            return isTopEquivalent;
        }


        public boolean isTopEquivalent(OWLDescription desc, Collection<? extends OWLEntity> signature,
                                       LocalityClass localityCls) {
            this.localityCls = localityCls;
            this.signature = signature;
            desc.accept(this);
            return isTopEquivalent;
        }


        public void setBottomEvaluator(BottomEquivalenceEvaluator evaluator) {
            this.bottomEvaluator = evaluator;
        }


        public void visit(OWLClass desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                    isTopEquivalent = desc.isOWLThing();
                    break;
                case TOP_BOTTOM:
                case TOP_TOP:
                    isTopEquivalent = desc.isOWLThing() || (!desc.isOWLNothing() && !signature.contains(desc));
                    break;
            }
        }


        public void visit(OWLDataAllRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isTopEquivalent = !signature.contains(desc.getProperty().asOWLDataProperty());
                    break;
                case TOP_TOP:
                    // FIXME: This ignores TOP_DATA case
                    // TS: But there is no TOP_DATA case (?)
                    isTopEquivalent = false;
                    break;
            }
        }


        // BUGFIX: (TS) Added the case where this is top-equiv (including n==0).
        public void visit(OWLDataExactCardinalityRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isTopEquivalent = (desc.getCardinality() == 0)
                            && (!signature.contains(desc.getProperty().asOWLDataProperty()));
                    break;
                case TOP_TOP:
                    isTopEquivalent = false;
                    break;
            }
        }


        // (TS) No special handling for n==0 required.
        public void visit(OWLDataMaxCardinalityRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isTopEquivalent = !signature.contains(desc.getProperty().asOWLDataProperty());
                    break;
                case TOP_TOP:
                    isTopEquivalent = false;
                    break;
            }
        }


        // BUGFIX: (TS) A data min card restriction is top-equiv iff the cardinality is 0.
        // TODO: (TS) If the filler can never be empty, then the TOP_TOP case can be bottom-equiv for n>0.
        public void visit(OWLDataMinCardinalityRestriction desc) {
            isTopEquivalent = (desc.getCardinality() == 0);
        }


        // BUGFIX: (TS) A data some restriction is never top-equivalent
        public void visit(OWLDataSomeRestriction desc) {
            isTopEquivalent = false;
        }


        // BUGFIX: (TS) A data value restriction is never top-equivalent 
        public void visit(OWLDataValueRestriction desc) {
            isTopEquivalent = false;
        }


        public void visit(OWLObjectAllRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isTopEquivalent = !signature.contains(desc.getProperty().getNamedProperty()) || isTopEquivalent(desc.getFiller());
                    break;
                case TOP_TOP:
                    isTopEquivalent = isTopEquivalent(desc.getFiller());
                    break;
            }
        }


        public void visit(OWLObjectComplementOf desc) {
            isTopEquivalent = bottomEvaluator.isBottomEquivalent(desc.getOperand(), signature, localityCls);
        }


        // BUGFIX: (TS) added the cases where this is top-equiv, including n==0
        public void visit(OWLObjectExactCardinalityRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isTopEquivalent = (desc.getCardinality() == 0)
                            && ((!signature.contains(desc.getProperty().getNamedProperty()))
                            || bottomEvaluator.isBottomEquivalent(desc.getFiller(), signature, localityCls));
                    break;
                case TOP_TOP:
                    isTopEquivalent = (desc.getCardinality() == 0)
                            && bottomEvaluator.isBottomEquivalent(desc.getFiller(), signature, localityCls);
                    break;
            }
        }


        public void visit(OWLObjectIntersectionOf desc) {
            for (OWLDescription conj : desc.getOperands()) {
                if (!isTopEquivalent(conj)) {
                    isTopEquivalent = false;
                    return;
                }
            }
            isTopEquivalent = true;
        }


        // BUGFIX: (TS) Added the case of a bottom-equivalent filler to both conditions.
        //              The n==0 case doesn't affect top-equivalence of this type of restriction.
        public void visit(OWLObjectMaxCardinalityRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isTopEquivalent = (!signature.contains(desc.getProperty().getNamedProperty()))
                            || bottomEvaluator.isBottomEquivalent(desc.getFiller(), signature, localityCls);
                    break;
                case TOP_TOP:
                    isTopEquivalent =
                            bottomEvaluator.isBottomEquivalent(desc.getFiller(), signature, localityCls);
                    break;
            }
        }


        // BUGFIX: (TS) Added the case n==0; repaired TOP_TOP condition 
        public void visit(OWLObjectMinCardinalityRestriction desc) {
            int card = desc.getCardinality();
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isTopEquivalent = (card == 0);
                    break;
                case TOP_TOP:
//                    isTopEquivalent = !signature.contains(desc.getProperty().getNamedProperty()) && (desc.getCardinality() <= 1);
                    isTopEquivalent = (card == 0)
                            || ((card > 0) && (!signature.contains(desc.getProperty().getNamedProperty()))
                            && (isTopEquivalent(desc.getFiller())));
                    break;
            }
        }


        public void visit(OWLObjectOneOf desc) {
            isTopEquivalent = false;
        }


        public void visit(OWLObjectSelfRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isTopEquivalent = false;
                    break;
                case TOP_TOP:
                    isTopEquivalent = !signature.contains(desc.getProperty().getNamedProperty());
                    break;
            }
        }


        // BUGFIX (TS): added ".getNamedProperty()"
        public void visit(OWLObjectSomeRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isTopEquivalent = false;
                    break;
                case TOP_TOP:
                    isTopEquivalent = !signature.contains(desc.getProperty().getNamedProperty()) && isTopEquivalent(desc.getFiller());
                    break;
            }
        }


        public void visit(OWLObjectUnionOf desc) {
            for (OWLDescription conj : desc.getOperands()) {
                if (isTopEquivalent(conj)) {
                    isTopEquivalent = true;
                    return;
                }
            }
            isTopEquivalent = false;
        }


        public void visit(OWLObjectValueRestriction desc) {
            switch (localityCls) {
                case BOTTOM_BOTTOM:
                case TOP_BOTTOM:
                    isTopEquivalent = false;
                    break;
                case TOP_TOP:
                    isTopEquivalent = !signature.contains(desc.getProperty().getNamedProperty());
                    break;
            }
        }
    }


    /**
     * True if the axiom is syntactically local w.r.t. given signature
     */
    public boolean isLocal(OWLAxiom axiom, Set<? extends OWLEntity> signature) {
        return axiomVisitor.isLocal(axiom, signature);
    }
}
