package org.semanticweb.owl.profiles;

import java.net.URI;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.semanticweb.owl.model.OWLAntiSymmetricObjectPropertyAxiom;
import org.semanticweb.owl.model.OWLAxiom;
import org.semanticweb.owl.model.OWLAxiomAnnotationAxiom;
import org.semanticweb.owl.model.OWLAxiomVisitorEx;
import org.semanticweb.owl.model.OWLClass;
import org.semanticweb.owl.model.OWLClassAssertionAxiom;
import org.semanticweb.owl.model.OWLDataAllRestriction;
import org.semanticweb.owl.model.OWLDataComplementOf;
import org.semanticweb.owl.model.OWLDataExactCardinalityRestriction;
import org.semanticweb.owl.model.OWLDataMaxCardinalityRestriction;
import org.semanticweb.owl.model.OWLDataMinCardinalityRestriction;
import org.semanticweb.owl.model.OWLDataOneOf;
import org.semanticweb.owl.model.OWLDataPropertyAssertionAxiom;
import org.semanticweb.owl.model.OWLDataPropertyDomainAxiom;
import org.semanticweb.owl.model.OWLDataPropertyRangeAxiom;
import org.semanticweb.owl.model.OWLDataRangeFacetRestriction;
import org.semanticweb.owl.model.OWLDataRangeRestriction;
import org.semanticweb.owl.model.OWLDataSomeRestriction;
import org.semanticweb.owl.model.OWLDataSubPropertyAxiom;
import org.semanticweb.owl.model.OWLDataType;
import org.semanticweb.owl.model.OWLDataValueRestriction;
import org.semanticweb.owl.model.OWLDataVisitorEx;
import org.semanticweb.owl.model.OWLDeclarationAxiom;
import org.semanticweb.owl.model.OWLDescription;
import org.semanticweb.owl.model.OWLDescriptionVisitorEx;
import org.semanticweb.owl.model.OWLDifferentIndividualsAxiom;
import org.semanticweb.owl.model.OWLDisjointClassesAxiom;
import org.semanticweb.owl.model.OWLDisjointDataPropertiesAxiom;
import org.semanticweb.owl.model.OWLDisjointObjectPropertiesAxiom;
import org.semanticweb.owl.model.OWLDisjointUnionAxiom;
import org.semanticweb.owl.model.OWLEntityAnnotationAxiom;
import org.semanticweb.owl.model.OWLEquivalentClassesAxiom;
import org.semanticweb.owl.model.OWLEquivalentDataPropertiesAxiom;
import org.semanticweb.owl.model.OWLEquivalentObjectPropertiesAxiom;
import org.semanticweb.owl.model.OWLFunctionalDataPropertyAxiom;
import org.semanticweb.owl.model.OWLFunctionalObjectPropertyAxiom;
import org.semanticweb.owl.model.OWLImportsDeclaration;
import org.semanticweb.owl.model.OWLInverseFunctionalObjectPropertyAxiom;
import org.semanticweb.owl.model.OWLInverseObjectPropertiesAxiom;
import org.semanticweb.owl.model.OWLIrreflexiveObjectPropertyAxiom;
import org.semanticweb.owl.model.OWLNegativeDataPropertyAssertionAxiom;
import org.semanticweb.owl.model.OWLNegativeObjectPropertyAssertionAxiom;
import org.semanticweb.owl.model.OWLObjectAllRestriction;
import org.semanticweb.owl.model.OWLObjectComplementOf;
import org.semanticweb.owl.model.OWLObjectExactCardinalityRestriction;
import org.semanticweb.owl.model.OWLObjectIntersectionOf;
import org.semanticweb.owl.model.OWLObjectMaxCardinalityRestriction;
import org.semanticweb.owl.model.OWLObjectMinCardinalityRestriction;
import org.semanticweb.owl.model.OWLObjectOneOf;
import org.semanticweb.owl.model.OWLObjectPropertyAssertionAxiom;
import org.semanticweb.owl.model.OWLObjectPropertyChainSubPropertyAxiom;
import org.semanticweb.owl.model.OWLObjectPropertyDomainAxiom;
import org.semanticweb.owl.model.OWLObjectPropertyRangeAxiom;
import org.semanticweb.owl.model.OWLObjectSelfRestriction;
import org.semanticweb.owl.model.OWLObjectSomeRestriction;
import org.semanticweb.owl.model.OWLObjectSubPropertyAxiom;
import org.semanticweb.owl.model.OWLObjectUnionOf;
import org.semanticweb.owl.model.OWLObjectValueRestriction;
import org.semanticweb.owl.model.OWLOntology;
import org.semanticweb.owl.model.OWLOntologyAnnotationAxiom;
import org.semanticweb.owl.model.OWLOntologyManager;
import org.semanticweb.owl.model.OWLReflexiveObjectPropertyAxiom;
import org.semanticweb.owl.model.OWLSameIndividualsAxiom;
import org.semanticweb.owl.model.OWLSubClassAxiom;
import org.semanticweb.owl.model.OWLSymmetricObjectPropertyAxiom;
import org.semanticweb.owl.model.OWLTransitiveObjectPropertyAxiom;
import org.semanticweb.owl.model.OWLTypedConstant;
import org.semanticweb.owl.model.OWLUntypedConstant;
import org.semanticweb.owl.model.SWRLRule;
import org.semanticweb.owl.vocab.OWLRDFVocabulary;
import org.semanticweb.owl.vocab.XSDVocabulary;

public class QLProfile implements OWLProfile {

	private static class QLAxiomChecker implements
			OWLAxiomVisitorEx<AxiomNotAllowed<? extends OWLAxiom>> {

		private final QLDataChecker dataChecker;
		private final QLSubClassChecker subChecker;
		private final QLSuperClassChecker superChecker;

		public QLAxiomChecker() {
			dataChecker = new QLDataChecker();
			subChecker = new QLSubClassChecker();
			superChecker = new QLSuperClassChecker(subChecker);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLAntiSymmetricObjectPropertyAxiom axiom) {
			return new AxiomNotAllowed<OWLAntiSymmetricObjectPropertyAxiom>(
					axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLAxiomAnnotationAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLClassAssertionAxiom axiom) {
			OWLDescription desc = axiom.getDescription();
			if (desc instanceof OWLClass)
				return null;
			else
				return new AxiomNotAllowed<OWLClassAssertionAxiom>(
						new DescriptionNotAllowed(desc), axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLDataPropertyAssertionAxiom axiom) {
			DataRangeNotAllowed cause = axiom.getObject().accept(dataChecker);
			if (cause != null)
				return new AxiomNotAllowed<OWLDataPropertyAssertionAxiom>(
						cause, axiom);

			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLDataPropertyDomainAxiom axiom) {
			DescriptionNotAllowed cause = axiom.getDomain()
					.accept(superChecker);
			if (cause != null)
				return new AxiomNotAllowed<OWLDataPropertyDomainAxiom>(cause,
						axiom);

			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLDataPropertyRangeAxiom axiom) {
			DataRangeNotAllowed cause = axiom.getRange().accept(dataChecker);
			if (cause != null)
				return new AxiomNotAllowed<OWLDataPropertyRangeAxiom>(cause,
						axiom);

			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLDataSubPropertyAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLDeclarationAxiom axiom) {
			// FIXME: Should prevent declaring unacceptable data ranges
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLDifferentIndividualsAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLDisjointClassesAxiom axiom) {
			for (OWLDescription desc : axiom.getDescriptions()) {
				DescriptionNotAllowed cause = desc.accept(subChecker);
				if (cause != null)
					return new AxiomNotAllowed<OWLDisjointClassesAxiom>(cause,
							axiom);
			}
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLDisjointDataPropertiesAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLDisjointObjectPropertiesAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLDisjointUnionAxiom axiom) {
			return new AxiomNotAllowed<OWLDisjointUnionAxiom>(axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLEntityAnnotationAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLEquivalentClassesAxiom axiom) {
			for (OWLDescription desc : axiom.getDescriptions()) {
				DescriptionNotAllowed cause = desc.accept(subChecker);
				if (cause != null)
					return new AxiomNotAllowed<OWLEquivalentClassesAxiom>(
							cause, axiom);
			}
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLEquivalentDataPropertiesAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLEquivalentObjectPropertiesAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLFunctionalDataPropertyAxiom axiom) {
			return new AxiomNotAllowed<OWLFunctionalDataPropertyAxiom>(axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLFunctionalObjectPropertyAxiom axiom) {
			return new AxiomNotAllowed<OWLFunctionalObjectPropertyAxiom>(axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLImportsDeclaration axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLInverseFunctionalObjectPropertyAxiom axiom) {
			return new AxiomNotAllowed<OWLInverseFunctionalObjectPropertyAxiom>(
					axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLInverseObjectPropertiesAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLIrreflexiveObjectPropertyAxiom axiom) {
			return new AxiomNotAllowed<OWLIrreflexiveObjectPropertyAxiom>(axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLNegativeDataPropertyAssertionAxiom axiom) {
			return new AxiomNotAllowed<OWLNegativeDataPropertyAssertionAxiom>(
					axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLNegativeObjectPropertyAssertionAxiom axiom) {
			return new AxiomNotAllowed<OWLNegativeObjectPropertyAssertionAxiom>(
					axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLObjectPropertyAssertionAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLObjectPropertyChainSubPropertyAxiom axiom) {
			return new AxiomNotAllowed<OWLObjectPropertyChainSubPropertyAxiom>(
					axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLObjectPropertyDomainAxiom axiom) {
			DescriptionNotAllowed cause = axiom.getDomain()
					.accept(superChecker);
			if (cause != null)
				return new AxiomNotAllowed<OWLObjectPropertyDomainAxiom>(cause,
						axiom);

			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLObjectPropertyRangeAxiom axiom) {
			DescriptionNotAllowed cause = axiom.getRange().accept(superChecker);
			if (cause != null)
				return new AxiomNotAllowed<OWLObjectPropertyRangeAxiom>(cause,
						axiom);

			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLObjectSubPropertyAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLOntologyAnnotationAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLReflexiveObjectPropertyAxiom axiom) {
			return new AxiomNotAllowed<OWLReflexiveObjectPropertyAxiom>(axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLSameIndividualsAxiom axiom) {
			return new AxiomNotAllowed<OWLSameIndividualsAxiom>(axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(OWLSubClassAxiom axiom) {
			DescriptionNotAllowed cause = axiom.getSubClass()
					.accept(subChecker);
			if (cause != null)
				return new AxiomNotAllowed<OWLSubClassAxiom>(cause, axiom);
			cause = axiom.getSuperClass().accept(superChecker);
			if (cause != null)
				return new AxiomNotAllowed<OWLSubClassAxiom>(cause, axiom);

			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLSymmetricObjectPropertyAxiom axiom) {
			return null;
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(
				OWLTransitiveObjectPropertyAxiom axiom) {
			return new AxiomNotAllowed<OWLTransitiveObjectPropertyAxiom>(axiom);
		}

		public AxiomNotAllowed<? extends OWLAxiom> visit(SWRLRule rule) {
			return new AxiomNotAllowed<SWRLRule>(rule);
		}

	}

	private static class QLDataChecker implements
			OWLDataVisitorEx<DataRangeNotAllowed> {
		
		private static final Set<URI> allowedDatatypes;
		
		static {
			Set<URI> s = new HashSet<URI>();
			s.add( OWLRDFVocabulary.RDF_XML_LITERAL.getURI() );
			s.add( OWLRDFVocabulary.RDFS_LITERAL.getURI() );
			s.add( XSDVocabulary.DECIMAL.getURI() );
			s.add( XSDVocabulary.INTEGER.getURI() );
			s.add( XSDVocabulary.NON_NEGATIVE_INTEGER.getURI() );
			s.add( XSDVocabulary.STRING.getURI() );
			s.add( XSDVocabulary.NORMALIZED_STRING.getURI() );
			s.add( XSDVocabulary.TOKEN.getURI() );
			s.add( XSDVocabulary.NAME.getURI() );
			s.add( XSDVocabulary.NCNAME.getURI() );
			s.add( XSDVocabulary.NMTOKEN.getURI() );
			s.add( XSDVocabulary.HEX_BINARY.getURI() );
			s.add( XSDVocabulary.BASE_64_BINARY.getURI() );
			s.add( XSDVocabulary.ANY_URI.getURI() );
			s.add( XSDVocabulary.DATE_TIME.getURI() );
			allowedDatatypes = Collections.unmodifiableSet( s );
		}

		public DataRangeNotAllowed visit(OWLDataComplementOf node) {
			return new DataRangeNotAllowed(node);
		}

		public DataRangeNotAllowed visit(OWLDataOneOf node) {
			return new DataRangeNotAllowed(node);
		}

		public DataRangeNotAllowed visit(OWLDataRangeFacetRestriction node) {
			throw new IllegalStateException();
		}

		public DataRangeNotAllowed visit(OWLDataRangeRestriction node) {
			return new DataRangeNotAllowed(node);
		}

		public DataRangeNotAllowed visit(OWLDataType node) {
			return allowedDatatypes.contains( node.getURI() )
				? null
				: new DataRangeNotAllowed( node );
		}

		public DataRangeNotAllowed visit(OWLTypedConstant node) {
			return node.getDataType().accept(this);
		}

		public DataRangeNotAllowed visit(OWLUntypedConstant node) {
			return null;
		}
	};

	private static class QLSubClassChecker implements
			OWLDescriptionVisitorEx<DescriptionNotAllowed> {

		public DescriptionNotAllowed visit(OWLClass desc) {
			return null;
		}

		public DescriptionNotAllowed visit(OWLDataAllRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(
				OWLDataExactCardinalityRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLDataMaxCardinalityRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLDataMinCardinalityRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLDataSomeRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLDataValueRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectAllRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectComplementOf desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(
				OWLObjectExactCardinalityRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectIntersectionOf desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(
				OWLObjectMaxCardinalityRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(
				OWLObjectMinCardinalityRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectOneOf desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectSelfRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectSomeRestriction desc) {
			OWLDescription filler = desc.getFiller();
			if (filler.isOWLThing())
				return null;
			else
				return new DescriptionNotAllowed(new DescriptionNotAllowed(
						filler), desc);
		}

		public DescriptionNotAllowed visit(OWLObjectUnionOf desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectValueRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}
	};

	private static class QLSuperClassChecker implements
			OWLDescriptionVisitorEx<DescriptionNotAllowed> {

		final private QLSubClassChecker subChecker;

		public QLSuperClassChecker(QLSubClassChecker subChecker) {
			this.subChecker = subChecker;
		}

		public DescriptionNotAllowed visit(OWLClass desc) {
			return null;
		}

		public DescriptionNotAllowed visit(OWLDataAllRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(
				OWLDataExactCardinalityRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLDataMaxCardinalityRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLDataMinCardinalityRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLDataSomeRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLDataValueRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectAllRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectComplementOf desc) {
			DescriptionNotAllowed cause = desc.getOperand().accept(subChecker);
			if (cause != null)
				return new DescriptionNotAllowed(cause, desc);

			return null;
		}

		public DescriptionNotAllowed visit(
				OWLObjectExactCardinalityRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectIntersectionOf desc) {
			for (OWLDescription d : desc.getOperands()) {
				DescriptionNotAllowed cause = d.accept(this);
				if (cause != null)
					return new DescriptionNotAllowed(cause, desc);
			}
			return null;
		}

		public DescriptionNotAllowed visit(
				OWLObjectMaxCardinalityRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(
				OWLObjectMinCardinalityRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectOneOf desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectSelfRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectSomeRestriction desc) {
			OWLDescription filler = desc.getFiller();
			if (filler instanceof OWLClass)
				return null;
			else
				return new DescriptionNotAllowed(new DescriptionNotAllowed(
						filler), desc);
		}

		public DescriptionNotAllowed visit(OWLObjectUnionOf desc) {
			return new DescriptionNotAllowed(desc);
		}

		public DescriptionNotAllowed visit(OWLObjectValueRestriction desc) {
			return new DescriptionNotAllowed(desc);
		}
	};

	public OWLProfileReport checkOntology(OWLOntology ontology,
			OWLOntologyManager manager) {
		QLAxiomChecker checker = new QLAxiomChecker();
		Set<ConstructNotAllowed> disallowedConstructs = new HashSet<ConstructNotAllowed>();
		for (OWLAxiom ax : ontology.getLogicalAxioms()) {
			AxiomNotAllowed<?> cause = ax.accept(checker);
			if (cause != null) {
				disallowedConstructs.add(cause);
			}
		}
		return new OWLProfileReport(this, ontology.getURI(),
				disallowedConstructs);
	}

	public String getName() {
		return "QL";
	}

}
