package org.coode.owl.latex;

import org.semanticweb.owl.model.*;
import org.semanticweb.owl.util.CollectionFactory;
import org.semanticweb.owl.util.ShortFormProvider;
import org.semanticweb.owl.util.SimpleShortFormProvider;

import java.util.*;
/*
 * Copyright (C) 2007, University of Manchester
 *
 * Modifications to the initial code base are copyright of their
 * respective authors, or their employers as appropriate.  Authorship
 * of the modifications may be determined from the ChangeLog placed at
 * the end of this file.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * Author: Matthew Horridge<br>
 * The University Of Manchester<br>
 * Medical Informatics Group<br>
 * Date: 15-Jun-2006<br><br>
 */
public class LatexObjectVisitor implements OWLObjectVisitor {

    public static final String AND = "\\sqcap";

    public static final String OR = "\\sqcup";

    public static final String NOT = "\\lnot";

    public static final String ALL = "\\forall\\,";

    public static final String SOME = "\\exists\\,";

    public static final String MIN = "\\mathnormal{\\geq}\\,";

    public static final String MAX = "\\mathnormal{\\leq}\\,";

    public static final String EQUAL = "\\mathnormal{=}\\,";

    public static final String SUBCLASS = "\\sqsubseteq";

    public static final String EQUIV = "\\equiv";

    public static final String NOT_EQUAL = "\\not=";

    public static final String TOP = "\\top";

    public static final String BOTTOM = "\\bot";

    public static final String SELF = "\\self";

    public static final String CIRC = "\\circ";


    private OWLObject subject;

    private LatexWriter writer;

    private boolean prettyPrint = true;

    private OWLDataFactory df;

    private ShortFormProvider shortFormProvider;


    public LatexObjectVisitor(LatexWriter writer, OWLDataFactory df) {
        this.writer = writer;
        this.df = df;
        shortFormProvider = new SimpleShortFormProvider();
        subject = df.getOWLThing();
    }


    public void setSubject(OWLObject subject) {
        this.subject = subject;
    }


    public void setShortFormProvider(ShortFormProvider shortFormProvder) {
        this.shortFormProvider = shortFormProvder;
    }


    private void writeSpace() {
        writer.writeSpace();
    }

    private void write(int i) {
        writer.write(Integer.toString(i));
        writer.write("\\,");
    }

    private void write(Object o) {
        writer.write(o);
    }


    private void writeOpenBrace() {
        writer.writeOpenBrace();
    }


    private void writeCloseBrace() {
        writer.writeCloseBrace();
    }


    public boolean isPrettyPrint() {
        return prettyPrint;
    }


    public void setPrettyPrint(boolean prettyPrint) {
        this.prettyPrint = prettyPrint;
    }


    public void visit(OWLObjectIntersectionOf node) {
        for (Iterator<OWLDescription> it = node.getOperands().iterator(); it.hasNext();) {
            it.next().accept(this);
            if (it.hasNext()) {
                writeSpace();
                write(AND);
                writeSpace();
            }
        }
    }

    private void writeQuantified(OWLQuantifiedRestriction quantifiedRestriction, String quantifier) {
        write(quantifier);
        writeSpace();
        quantifiedRestriction.getProperty().accept(this);
        if (quantifiedRestriction.isQualified()) {
            write(".");
            writeNested(quantifiedRestriction.getFiller());
        }
    }


    private void writeCardinality(OWLCardinalityRestriction cardinalityRestriction, String quantifier) {
        write(quantifier);
        writeSpace();
        write(cardinalityRestriction.getCardinality());
        cardinalityRestriction.getProperty().accept(this);
        if(cardinalityRestriction.isQualified()) {
            write(".");
            writeNested(cardinalityRestriction.getFiller());
        }
    }

    public void visit(OWLDataAllRestriction node) {
        writeQuantified(node, ALL);
    }


    public void visit(OWLDataExactCardinalityRestriction desc) {
        writeCardinality(desc, EQUAL);
    }


    public void visit(OWLDataMaxCardinalityRestriction desc) {
        writeCardinality(desc, MAX);
    }


    public void visit(OWLDataMinCardinalityRestriction desc) {
        writeCardinality(desc, MIN);
    }


    public void visit(OWLDataSomeRestriction node) {
        writeQuantified(node, SOME);
    }


    public void visit(OWLDataValueRestriction node) {
        node.asSomeValuesFrom().accept(this);
    }


    public void visit(OWLObjectAllRestriction node) {
        writeQuantified(node, ALL);
    }


    public void visit(OWLObjectExactCardinalityRestriction desc) {
        writeCardinality(desc, EQUAL);
    }


    public void visit(OWLObjectMaxCardinalityRestriction desc) {
        writeCardinality(desc, MAX);
    }


    public void visit(OWLObjectMinCardinalityRestriction desc) {
        writeCardinality(desc, MIN);
    }


    public void visit(OWLObjectSomeRestriction node) {
        writeQuantified(node, SOME);
    }


    public void visit(OWLObjectValueRestriction node) {
        node.asSomeValuesFrom().accept(this);
    }


    public void visit(OWLObjectComplementOf node) {
        write(NOT);
        writeNested(node.getOperand());
    }


    public void visit(OWLObjectUnionOf node) {
        for (Iterator<OWLDescription> it = node.getOperands().iterator(); it.hasNext();) {
            it.next().accept(this);
            if (it.hasNext()) {
                writeSpace();
                write(OR);
                writeSpace();
            }
        }
    }


    public void visit(OWLClass node) {
        if(node.isOWLThing()) {
            write(TOP);
        }
        else if(node.isOWLNothing()) {
            write(BOTTOM);
        }
        else {
            write("\\class{");
            write(escapeName(shortFormProvider.getShortForm(node)));
            write("}");
        }
    }


    public void visit(OWLObjectOneOf node) {
        for (Iterator<OWLIndividual> it = node.getIndividuals().iterator(); it.hasNext();) {
            writeOpenBrace();
            it.next().accept(this);
            writeCloseBrace();
            if (it.hasNext()) {
                writeSpace();
                write(OR);
                writeSpace();
            }
        }
    }


    public void visit(OWLDataProperty entity) {
        write("\\dataproperty{");
        write(escapeName(shortFormProvider.getShortForm(entity)));
        write("}");
    }


    public void visit(OWLObjectProperty entity) {
        write("\\objectproperty{");
        write(escapeName(shortFormProvider.getShortForm(entity)));
        write("}");
    }


    public void visit(OWLIndividual entity) {
        write("\\individual{");
        write(escapeName(shortFormProvider.getShortForm(entity)));
        write("}");
    }


    public void visit(OWLObjectSelfRestriction desc) {
        write(SOME);
        writeSpace();
        desc.getProperty().accept(this);
        writeSpace();
        write(SELF);
    }


    public void visit(OWLDisjointClassesAxiom axiom) {
        if (axiom.getDescriptions().size() != 2) {
            for (OWLDescription left : axiom.getDescriptions()) {
                for (OWLDescription right : axiom.getDescriptions()) {
                    if (left != right) {
                        if (left.equals(subject)) {
                            left.accept(this);
                            writeSpace();
                            write(SUBCLASS);
                            writeSpace();
                            write(NOT);
                            writeSpace();
                            right.accept(this);
                        }
                        else {
                            right.accept(this);
                            writeSpace();
                            write(SUBCLASS);
                            writeSpace();
                            write(NOT);
                            writeSpace();
                            left.accept(this);
                        }
                        writer.writeNewLine();
                    }
                }
            }
        }
        else {
            Iterator<OWLDescription> it = axiom.getDescriptions().iterator();
            OWLDescription descA = it.next();
            OWLDescription descB = it.next();
            OWLDescription lhs;
            OWLDescription rhs;
            if(descA.equals(subject)) {
                lhs = descA;
                rhs = descB;
            }
            else {
                lhs = descB;
                rhs = descA;
            }
            lhs.accept(this);
            writeSpace();
            write(SUBCLASS);
            writeSpace();
            write(NOT);
            writeSpace();
            rhs.accept(this);
        }
    }


    public void visit(OWLEquivalentClassesAxiom axiom) {
        if (axiom.getDescriptions().size() > 2) {
            Set<Set<OWLDescription>> rendered = new HashSet<Set<OWLDescription>>();
            for (OWLDescription left : axiom.getDescriptions()) {
                for (OWLDescription right : axiom.getDescriptions()) {
                    if (left != right) {
                        Set<OWLDescription> cur = CollectionFactory.createSet(left, right);
                        if(!rendered.contains(cur)) {
                            rendered.add(cur);
                            left.accept(this);
                            writeSpace();
                            write(EQUIV);
                            writeSpace();
                            right.accept(this);
                        }

                    }
                }
            }
        }
        else if(axiom.getDescriptions().size() == 2) {
            Iterator<OWLDescription> it = axiom.getDescriptions().iterator();
            OWLDescription descA = it.next();
            OWLDescription descB = it.next();
            OWLDescription lhs;
            OWLDescription rhs;
            if(subject.equals(descA)) {
                lhs = descA;
                rhs = descB;
            }
            else {
                lhs = descB;
                rhs = descA;
            }
            lhs.accept(this);
            writeSpace();
            write(EQUIV);
            writeSpace();
            rhs.accept(this);
        }
    }


    public void visit(OWLSubClassAxiom axiom) {
        this.setPrettyPrint(false);
        axiom.getSubClass().accept(this);
        writeSpace();
        write(SUBCLASS);
        writeSpace();
        axiom.getSuperClass().accept(this);
        writeSpace();
        this.setPrettyPrint(true);
    }


    public void visit(OWLClassAssertionAxiom axiom) {
        if(axiom.getDescription().isAnonymous()) {
            write("(");
        }
        axiom.getDescription().accept(this);
        if(axiom.getDescription().isAnonymous()) {
            write("(");
        }
        write("(");
        axiom.getIndividual().accept(this);
        write(")");
    }


    public void visit(OWLAntiSymmetricObjectPropertyAxiom axiom) {
        df.getOWLDisjointObjectPropertiesAxiom(axiom.getProperty(), axiom.getProperty().getInverseProperty());
    }


    public void visit(OWLAxiomAnnotationAxiom axiom) {
    }


    public void visit(OWLDataPropertyAssertionAxiom axiom) {
        axiom.getProperty().accept(this);
        write("(");
        axiom.getSubject().accept(this);
        write(", ");
        axiom.getObject().accept(this);
        write(")");
    }


    public void visit(OWLDataPropertyDomainAxiom axiom) {
        axiom.asSubClassAxiom().accept(this);
    }


    public void visit(OWLDataPropertyRangeAxiom axiom) {
        axiom.asSubClassAxiom().accept(this);
    }


    public void visit(OWLDataSubPropertyAxiom axiom) {
        axiom.getSubProperty().accept(this);
        write(SUBCLASS);
        axiom.getSuperProperty().accept(this);
    }


    public void visit(OWLDeclarationAxiom axiom) {
    }


    public void visit(OWLDifferentIndividualsAxiom axiom) {
        writePairwise(axiom.getIndividuals(), NOT_EQUAL);
    }


    public void visit(OWLDisjointDataPropertiesAxiom axiom) {
        writePairwiseDisjoint(axiom.getProperties());
    }


    public void visit(OWLDisjointObjectPropertiesAxiom axiom) {
        writePairwiseDisjoint(axiom.getProperties());
    }

    private void writePairwiseDisjoint(Set<? extends OWLObject> objects) {
        writePairwise(objects, SUBCLASS + NOT);
    }

    private void writePairwise(Set<? extends OWLObject> objects, String delim) {
        List<OWLObject> list = new ArrayList<OWLObject>(objects);
        for(int i = 0; i < list.size() - 1; i++) {
            for(int j = i + 1; j < list.size() ; j++) {
                OWLObject o1 = list.get(i);
                OWLObject o2 = list.get(j);
                if(o1.equals(subject)) {
                    o1.accept(this);
                    write(delim);
                    o2.accept(this);
                    write(" \\\\ ");
                }
                else {
                    o2.accept(this);
                    write(delim);
                    o1.accept(this);
                    write(" \\\\ ");
                }
            }
        }
    }


    public void visit(OWLDisjointUnionAxiom axiom) {
    }


    public void visit(OWLEntityAnnotationAxiom axiom) {
    }


    public void visit(OWLEquivalentDataPropertiesAxiom axiom) {
        writePairwise(axiom.getProperties(), EQUIV);
    }


    public void visit(OWLEquivalentObjectPropertiesAxiom axiom) {
        writePairwise(axiom.getProperties(), EQUIV);
    }


    public void visit(OWLFunctionalDataPropertyAxiom axiom) {
        write(TOP);
        writeSpace();
        write(SUBCLASS);
        writeSpace();
        df.getOWLDataMaxCardinalityRestriction(axiom.getProperty(), 1).accept(this);
    }


    public void visit(OWLFunctionalObjectPropertyAxiom axiom) {
        write(TOP);
        writeSpace();
        write(SUBCLASS);
        writeSpace();
        df.getOWLObjectMaxCardinalityRestriction(axiom.getProperty(), 1).accept(this);
    }


    public void visit(OWLImportsDeclaration axiom) {
    }


    public void visit(OWLInverseFunctionalObjectPropertyAxiom axiom) {
        write(TOP);
        writeSpace();
        write(SUBCLASS);
        writeSpace();
        OWLObjectPropertyExpression prop = df.getOWLObjectPropertyInverse(axiom.getProperty());
        df.getOWLObjectMaxCardinalityRestriction(prop, 1).accept(this);
    }


    public void visit(OWLInverseObjectPropertiesAxiom axiom) {
        write(axiom.getFirstProperty());
        writeSpace();
        write(EQUIV);
        writeSpace();
        write(axiom.getSecondProperty());
        write("^-");
    }


    public void visit(OWLIrreflexiveObjectPropertyAxiom axiom) {

    }


    public void visit(OWLNegativeDataPropertyAssertionAxiom axiom) {
        write(NOT);
        axiom.getProperty().accept(this);
        write("(");
        axiom.getSubject().accept(this);
        write(", ");
        axiom.getObject().accept(this);
        write(")");
    }


    public void visit(OWLNegativeObjectPropertyAssertionAxiom axiom) {
        write(NOT);
        axiom.getProperty().accept(this);
        write("(");
        axiom.getSubject().accept(this);
        write(", ");
        axiom.getObject().accept(this);
        write(")");
    }


    public void visit(OWLObjectPropertyAssertionAxiom axiom) {
        axiom.getProperty().accept(this);
        write("(");
        axiom.getSubject().accept(this);
        write(", ");
        axiom.getObject().accept(this);
        write(")");
    }


    public void visit(OWLObjectPropertyChainSubPropertyAxiom axiom) {
        for (Iterator<OWLObjectPropertyExpression> it = axiom.getPropertyChain().iterator(); it.hasNext();) {
            it.next().accept(this);
            if (it.hasNext()) {
                writeSpace();
                write(CIRC);
                writeSpace();
            }
        }
        writeSpace();
        write(SUBCLASS);
        writeSpace();
        axiom.getSuperProperty().accept(this);
    }


    public void visit(OWLObjectPropertyDomainAxiom axiom) {
        axiom.asSubClassAxiom().accept(this);
    }


    public void visit(OWLObjectPropertyRangeAxiom axiom) {
        axiom.asSubClassAxiom().accept(this);
    }


    public void visit(OWLObjectSubPropertyAxiom axiom) {
        axiom.getSubProperty().accept(this);
        writeSpace();
        write(SUBCLASS);
        writeSpace();
        axiom.getSuperProperty().accept(this);
    }


    public void visit(OWLOntologyAnnotationAxiom axiom) {
    }


    public void visit(OWLReflexiveObjectPropertyAxiom axiom) {
        axiom.asSubClassAxiom().accept(this);
    }


    public void visit(OWLSameIndividualsAxiom axiom) {
        writePairwise(axiom.getIndividuals(), EQUAL);
    }


    public void visit(OWLSymmetricObjectPropertyAxiom axiom) {
        axiom.getProperty().accept(this);
        writeSpace();
        write(EQUIV);
        writeSpace();
        axiom.getProperty().accept(this);
        write("^-");
    }


    public void visit(OWLTransitiveObjectPropertyAxiom axiom) {
        axiom.getProperty().accept(this);
        writeSpace();
        write(CIRC);
        writeSpace();
        axiom.getProperty().accept(this);
        writeSpace();
        write(SUBCLASS);
        writeSpace();
        axiom.getProperty().accept(this);

    }


    public void visit(SWRLRule rule) {
    }

    private void writeNested(OWLPropertyRange range) {
        if(range instanceof OWLDataRange) {
            range.accept(this);
        }
        else {
            writeNested((OWLDescription) range);
        }
    }


    private void writeNested(OWLDescription description) {
        openBracket(description);
        description.accept(this);
        closeBracket(description);
    }


    private void openBracket(OWLDescription description) {
        if (LatexBracketChecker.requiresBracket(description)) {
            write("(");
        }
    }


    private void closeBracket(OWLDescription description) {
        if (LatexBracketChecker.requiresBracket(description)) {
            write(")");
        }
    }


    private String escapeName(String name) {
        return name.replace("_", "\\_");
    }


    public void visit(OWLOntology ontology) {
    }


    public void visit(OWLObjectPropertyInverse property) {
        property.getInverse().accept(this);
        write("^-");
    }


    public void visit(OWLConstantAnnotation annotation) {

    }


    public void visit(OWLObjectAnnotation annotation) {
    }

    //////////////////////////////////////////////////////////////////////////////////////////////////


    public void visit(OWLDataComplementOf node) {
    }


    public void visit(OWLDataOneOf node) {
        for(Iterator<OWLConstant> it = node.getValues().iterator(); it.hasNext(); ) {
            writeOpenBrace();
            it.next().accept(this);
            writeCloseBrace();
            if(it.hasNext()) {
                writeSpace();
                write(OR);
                writeSpace();
            }
        }
    }


    public void visit(OWLDataRangeFacetRestriction node) {
    }


    public void visit(OWLDataRangeRestriction node) {
    }


    public void visit(OWLDataType node) {
        write("\\datatype{");
        write(shortFormProvider.getShortForm(node));
        write("}");
    }


    public void visit(OWLTypedConstant node) {
        writeConstant(node);
    }

    private void writeConstant(OWLConstant node) {
        write("\\constant{");
        write(node.getLiteral());
        write("}");
    }


    public void visit(OWLUntypedConstant node) {
        writeConstant(node);
    }

    //////////////////////////////////////////////////////////////////////////////////////////////


    public void visit(SWRLAtomConstantObject node) {
    }


    public void visit(SWRLAtomDVariable node) {
    }


    public void visit(SWRLAtomIndividualObject node) {
    }


    public void visit(SWRLAtomIVariable node) {
    }


    public void visit(SWRLBuiltInAtom node) {
    }


    public void visit(SWRLClassAtom node) {
    }


    public void visit(SWRLDataRangeAtom node) {
    }


    public void visit(SWRLDataValuedPropertyAtom node) {
    }


    public void visit(SWRLDifferentFromAtom node) {
    }


    public void visit(SWRLObjectPropertyAtom node) {
    }


    public void visit(SWRLSameAsAtom node) {
    }
}
