package uk.ac.manchester.cs.owl.modularity;

import org.semanticweb.owl.model.*;
import org.semanticweb.owl.modularity.OntologySegment;

import java.net.URI;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;

/**
 * Implementation of modules based on syntactic locality.
 *
 * @author Thomas Schneider
 * @author School of Computer Science
 * @author University of Manchester
 */
public class SyntacticLocalityModule implements OntologySegment {

    protected OWLOntology baseOntology;
    protected OWLOntologyManager baseOntologyManager;

    protected SyntacticLocalityModuleExtractor associatedExtractor;

    protected boolean[] axiomArray;

    protected Set<OWLAxiom> logicalAxioms;
    protected Set<OWLAxiom> nonLogicalAxioms;
    protected Set<OWLAxiom> axioms;

    protected ModuleType moduleType;

    protected Set<OWLEntity> seedSignature;
    protected Set<OWLEntity> moduleSignature;

    /**
     * Creates a new module that consists of the specified set of axioms and the array of Booleans.
     * The specified extractor, its associated ontology, and its module type will be associated
     * with this module.
     *
     * @param extractor         the extractor that created this module
     * @param logicalAxioms     the logical axioms in this module
     * @param nonLogicalAxioms  the non-logical axioms in this module
     * @param modArray          the Boolean representation of the set of axioms
     * @param seedSignature  the signature for which this module has been extracted
     * @param moduleSignature    the signature of this module
     */
    public SyntacticLocalityModule(SyntacticLocalityModuleExtractor extractor,
                                   Set<OWLAxiom> logicalAxioms, Set<OWLAxiom> nonLogicalAxioms, boolean[] modArray,
                                   Set<OWLEntity> seedSignature, Set<OWLEntity> moduleSignature
    ){
        this.associatedExtractor = extractor;
        this.baseOntology = extractor.getAssociatedOntology();
        this.baseOntologyManager = extractor.getAssociatedOntologyManager();
        this.logicalAxioms = new HashSet<OWLAxiom>(logicalAxioms);
        this.nonLogicalAxioms = new HashSet<OWLAxiom>(nonLogicalAxioms);
        this.axioms = new HashSet<OWLAxiom>(logicalAxioms);
        this.axioms.addAll(nonLogicalAxioms);
        this.axiomArray = modArray;
        this.moduleType = extractor.getModuleType();
        this.seedSignature = seedSignature;
        this.moduleSignature = moduleSignature;
    }

    /**
     * Returns the set of axioms that this module consists of.
     * @return the module as a set of axioms
     */
    public Set<OWLAxiom> getAxioms() {
        return axioms;
    }

    /**
     * Returns the set of logical axioms in this module.
     * @return all logical axioms in this module
     */
    public Set<OWLAxiom> getLogicalAxioms() {
        return logicalAxioms;
    }

    /**
     * Returns the set of non-logical axioms in this module.
     * @return all non-logical axioms in this module
     */
    public Set<OWLAxiom> getNonLogicalAxioms() {
        return nonLogicalAxioms;
    }

    /**
     * Returns the signature (set of entities) for which this module has been extracted.
     * @return the seed signature as a set of entities
     */
    public Set<OWLEntity> getSeedSignature() {
        return seedSignature;
    }

    /**
     * Returns the signature of this module, i.e. the set of entities that occur in it.
     * @return the signature as a set of entities
     */
    public Set<OWLEntity> getSignature() {
        return moduleSignature;
    }

    /**
     * Returns the signature of this module, i.e. the set of entities that occur in it.
     * Internally, this method simply calls getSignature().
     * Its purpose is to allow for a clear distinction between the seed signature and the signature
     * of the module.
     * @return the signature as a set of entities
     */
    public Set<OWLEntity> getModuleSignature() {
        return getSignature();
    }

    /**
     * Returns this module as a new ontology.
     * @param uri the URI for the new ontology
     * @return the new ontology, having the specified URI
     * @throws OWLOntologyCreationException  if the new ontology cannot be created
     * @throws OWLOntologyChangeException    if adding axioms to the new ontology fails
     */
    public OWLOntology asOWLOntology(URI uri) throws OWLOntologyCreationException, OWLOntologyChangeException {
        OWLOntology newOnt = baseOntologyManager.createOntology(uri);
        LinkedList<AddAxiom> addaxs = new LinkedList<AddAxiom>();
        for (OWLAxiom ax : axioms) {
            addaxs.add(new AddAxiom(newOnt, ax));
        }
        baseOntologyManager.applyChanges(addaxs);
		return newOnt;
    }
}
