package uk.ac.manchester.cs.owl.modularity;

import com.clarkparsia.modularity.locality.LocalityClass;
import com.clarkparsia.modularity.locality.SyntacticLocalityEvaluator;
import org.semanticweb.owl.model.*;
import org.semanticweb.owl.modularity.OntologySegmenter;

import java.net.URI;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Set;

/**
 * Implementation of module extraction based on syntactic locality.
 *
 * @author Thomas Schneider
 * @author School of Computer Science
 * @author University of Manchester
 */
public class SyntacticLocalityModuleExtractor implements OntologySegmenter {

    /**
     * Auxiliary inner class for the representation of the associated ontology and all its sub-ontologies as arrays of
     * axioms. Advantages: (1) quicker set manipulation operations; (2) storage of all referenced entities of an axiom once
     * this axiom is dealt with.
     *
     * @author Thomas Schneider
     * @author School of Computer Science
     * @author University of Manchester
     */
    protected class OntologyAxiomSet {

        /**
         * Array representing all axioms of the associated ontology.
         */
        protected OWLAxiom[] ax;


        /**
         * Creates a new OntologyAxiomSet from a given set of axioms without looking up the referenced entities.
         *
         * @param axs the set of axioms representing the ontology
         */
        public OntologyAxiomSet(Set<OWLAxiom> axs) {
            ax = axs.toArray(new OWLAxiom[axs.size()]);
        }


        /**
         * Returns the number of axioms in this set.
         *
         * @return the number of axioms in this set
         */
        public int size() {
            return ax.length;
        }


        /**
         * Returns some axiom from this set.
         *
         * @param i a number for an axiom
         * @return the i-th axiom in this set
         */
        public OWLAxiom getAxiom(int i) {
            return ax[i];
        }


        /**
         * Returns an array containing all axioms in this set.
         *
         * @return array containing all axioms in this set
         */
        public OWLAxiom[] getAllAxioms() {
            return ax;
        }


        /**
         * Returns the set of axioms that is represented by some array of Booleans.
         *
         * @param isIn an array of Booleans
         * @return the set of axioms represented by the specified array of Booleans
         */
        public Set<OWLAxiom> getAxiomSet(boolean[] isIn) {
            HashSet<OWLAxiom> gas = new HashSet<OWLAxiom>();
            for (int i = 0; i < isIn.length; i++) {
                if (isIn[i]) {
                    gas.add(ax[i]);
                }
            }
            return gas;
        }


        /**
         * Constructs an array of Booleans that represents a subset of this set. The subset either equals this set (if
         * init==true) or is the empty set (if init==false).
         *
         * @param init determines the initial value of the subset
         * @return array of Booleans representing the specified subset
         */
        public boolean[] getSubset(boolean init) {
            boolean[] subset = new boolean[ax.length];
            for (int i = 0; i < ax.length; i++)
                subset[i] = init;
            return subset;
        }


        /**
         * Clones an array of Booleans that represents a subset of this set.
         *
         * @param oldSubset an array representing the original subset
         * @return an array representing the new subset
         */
        public boolean[] cloneSubset(boolean[] oldSubset) {
            boolean[] newSubset = new boolean[ax.length];
            System.arraycopy(oldSubset, 0, newSubset, 0, ax.length);
            return newSubset;
        }


        public int subsetCardinality(boolean[] subset) {
            int card = 0;
            for (int i = 0; i < ax.length; i++) {
                if (subset[i])
                    card++;
            }
            return card;
        }


        /**
         * Transforms a subset of this set (represented by an array of Booleans) into a set of axioms.
         *
         * @param subset an array representing the subset
         * @return a set of axioms
         */
        public Set<OWLAxiom> toSet(boolean[] subset) {
            HashSet<OWLAxiom> axs = new HashSet<OWLAxiom>();
            for (int i = 0; i < ax.length; i++) {
                if (subset[i]) {
                    axs.add(ax[i]);
                }
            }
            return axs;
        }
    }


    /**
     * Type of module
     */
    protected ModuleType moduleType;

    /**
     * Represents the associated ontology.
     */
    protected OntologyAxiomSet ontologyAxiomSet;

    /**
     * Represents the ontology associated with this module extractor.
     */
    protected OWLOntology associatedOntology;
    
    /**
     * Represents the manager for the associated ontology.
     */
    protected OWLOntologyManager associatedOntologyManager;

    /**
     * Returns the ontology associated with this module extractor.
     * @return the ontology associated with this module extractor
     */
    public OWLOntology getAssociatedOntology() {
        return associatedOntology;
    }

    /**
     * Returns the manager for the ontology associated with this module extractor.
     * @return the manager for the ontology associated with this module extractor
     */
    public OWLOntologyManager getAssociatedOntologyManager() {
        return associatedOntologyManager;
    }


    /**
     * Creates a new module extractor for a subset of a given ontology, its manager, and a specified type of locality.
     *
     * @param man        the manager for the associated ontology
     * @param ont        the associated ontology
     * @param axs        the subset of the ontology as a set of axioms
     * @param moduleType the type of module this extractor will construct
     */
    public SyntacticLocalityModuleExtractor(OWLOntologyManager man, OWLOntology ont, Set<OWLAxiom> axs, ModuleType moduleType) {
        setModuleType(moduleType);

        associatedOntologyManager = man;
        associatedOntology = ont;
        ontologyAxiomSet = new OntologyAxiomSet(axs);
    }


    /**
     * Creates a new module extractor for a given ontology, its manager, and a specified type of locality.
     *
     * @param man        the manager for the associated ontology
     * @param ont        the associated ontology
     * @param moduleType the type of module this extractor will construct
     */
    public SyntacticLocalityModuleExtractor(OWLOntologyManager man, OWLOntology ont, ModuleType moduleType) {
        this(man, ont, ont.getAxioms(), moduleType);
    }


    /**
     * Changes the module type for this extractor without deleting the stored referenced entities.
     *
     * @param moduleType the new type of module
     */
    public void setModuleType(ModuleType moduleType) {
        this.moduleType = moduleType;
    }


    /**
     * Returns the module type for this extractor.
     *
     * @return module type for this extractor
     */
    public ModuleType getModuleType() {
        return moduleType;
    }


    /**
     * This auxiliary method extracts a module from a given sub-ontology of the associated ontology for a given signature
     * and locality type. The module will contain only logical axioms, no annotation or declaration axioms.
     * The sub-ontology and module are represented as arrays of Booleans.
     * <p/>
     * This method is (if necessary, iteratively) called by the public method extract.
     *
     * @param subOnt        an array of Booleans representing the sub-ontology
     * @param signature     the seed signature (set of entities) for the module; on return of the method, this will contain the signature of the module 
     * @param localityClass the type of locality
     * @return an array of Booleans representing the module
     */
    protected boolean[] extractLogicalAxioms(boolean[] subOnt, Set<OWLEntity> signature, LocalityClass localityClass) {
        boolean[] mod = ontologyAxiomSet.getSubset(false);
        boolean[] q2 = ontologyAxiomSet.cloneSubset(subOnt);

        SyntacticLocalityEvaluator sle = new SyntacticLocalityEvaluator(localityClass);

        boolean change = true;
        while (change) {
            change = false;
            for (int i = 0; i < q2.length; i = i + 1) {
                if (q2[i] && !sle.isLocal(ontologyAxiomSet.getAxiom(i), signature)) {
                    mod[i] = true;
                    q2[i] = false;
                    int oldSize = signature.size();
                    signature.addAll(ontologyAxiomSet.getAxiom(i).getSignature());
                    // only triggering a change when the signature has changed doesn't improve performance
                    if (signature.size() > oldSize) {
                        change = true;
                    }
                }
            }
        }
        return mod;
    }


    /**
     * This method returns all non-logical axioms (declaration and annotation axioms)
     * that are associated with the logical axioms in the specified module.
     *
     * @param logicalAxioms  set of logical axioms representing the original module
     * @param sig            a set of entities representing the signature of the original module
     * @return               a set of axioms representing the enriched module
     */
    protected Set<OWLAxiom> extractNonLogicalAxioms(Set<OWLAxiom> logicalAxioms, Set<OWLEntity> sig) {
        Set<OWLAxiom> allAxioms = new HashSet<OWLAxiom>(logicalAxioms);
        Set<OWLAxiom> nonLogicalAxioms = new HashSet<OWLAxiom>();

        //Adding all entity declaration axioms
        for (int i = 0; i < ontologyAxiomSet.size(); i++) {
            OWLAxiom axiom = ontologyAxiomSet.getAxiom(i);
            if (OWLDeclarationAxiom.class.isAssignableFrom(axiom.getClass())) {
                if (sig.contains(((OWLDeclarationAxiom) axiom).getEntity())) {
                    allAxioms.add(axiom);
                    nonLogicalAxioms.add(axiom);
                }
            }
        }

        // Adding all entity annotation axioms
        for (OWLEntity entity : sig) {
            Set<OWLAnnotationAxiom> annotationAxioms = entity.getAnnotationAxioms(associatedOntology);
            allAxioms.addAll(annotationAxioms);
            nonLogicalAxioms.addAll(annotationAxioms);
        }

        // Add all axiom annotation axioms recursively
        LinkedList<OWLAxiom> annotations = new LinkedList<OWLAxiom>();
        Iterator<OWLAxiom> iterator = allAxioms.iterator();
        while (iterator.hasNext()) {
            OWLAxiom axiom = iterator.next();
            annotations.addAll(axiom.getAnnotationAxioms(associatedOntology));
            while (!annotations.isEmpty()) {
                OWLAxiom first = annotations.remove();
                annotations.addAll(first.getAnnotationAxioms(associatedOntology));
                allAxioms.add(first);
                nonLogicalAxioms.add(first);
            }
        }

        return nonLogicalAxioms;
    }


    /**
     * Convenience method for extracting a lower-of-upper or upper-of-lower module in one go.
     * Only logical axioms are extracted.
     *
     * @param signature the seed signature (set of entities) for the module; on return of the method, this will contain the signature of the module
     * @param subOnt    an array of Booleans representing the sub-ontology
     * @param cls1      the locality class for the first extraction step
     * @param cls2      the locality class for the second extraction step
     * @return          a set of axioms representing the module 
     */
    protected boolean[] extractNestedLogicalModule(Set<OWLEntity> signature, boolean[] subOnt, LocalityClass cls1, LocalityClass cls2) {
        Set<OWLEntity> seedSig = new HashSet<OWLEntity>(signature);
        boolean[] preModule = extractLogicalAxioms(subOnt, seedSig, cls1);
        return extractLogicalAxioms(preModule, signature, cls2);
    }

    /**
     * Extracts a module from the associated ontology for a given signature and the associated module type.
     * The module will include annotation and declaration axioms for all entities and axioms in it.
     *
     * @param sig the seed signature (set of entities) for the module
     * @return the module
     */
    public SyntacticLocalityModule extractModule(Set<OWLEntity> sig) {
        boolean[] subOnt = ontologyAxiomSet.getSubset(true);
        boolean[] module;
        Set<OWLEntity> signature = new HashSet<OWLEntity>(sig);
        switch (moduleType) {
            case TOP: {
                module = extractLogicalAxioms(subOnt, signature, LocalityClass.TOP_TOP);
                break;
            }
            case BOT: {
                module = extractLogicalAxioms(subOnt, signature, LocalityClass.BOTTOM_BOTTOM);
                break;
            }
            case BOT_OF_TOP: {
                module = extractNestedLogicalModule(signature, subOnt, LocalityClass.TOP_TOP, LocalityClass.BOTTOM_BOTTOM);
                break;
            }
            case TOP_OF_BOT: {
                module = extractNestedLogicalModule(signature, subOnt, LocalityClass.BOTTOM_BOTTOM, LocalityClass.TOP_TOP);
                break;
            }
            case INTERSECTION: {
                Set<OWLEntity> signature1 = new HashSet<OWLEntity>(sig);
                boolean[] module1 = extractNestedLogicalModule(signature1, subOnt, LocalityClass.TOP_TOP, LocalityClass.BOTTOM_BOTTOM);
                Set<OWLEntity> signature2 = new HashSet<OWLEntity>(sig);
                boolean[] module2 = extractNestedLogicalModule(signature2, subOnt, LocalityClass.BOTTOM_BOTTOM, LocalityClass.TOP_TOP);

                module = new boolean[module1.length];
                for (int i = 0; i < module.length; i++){
                    module[i] = module1[i] && module2[i];
                }
                break;
            }
            case BEST: {
                Set<OWLEntity> signature1 = new HashSet<OWLEntity>(sig);
                boolean[] module1 = extractNestedLogicalModule(signature1, subOnt, LocalityClass.TOP_TOP, LocalityClass.BOTTOM_BOTTOM);
                Set<OWLEntity> signature2 = new HashSet<OWLEntity>(sig);
                boolean[] module2 = extractNestedLogicalModule(signature2, subOnt, LocalityClass.BOTTOM_BOTTOM, LocalityClass.TOP_TOP);

                if (ontologyAxiomSet.subsetCardinality(module1) < ontologyAxiomSet.subsetCardinality(module2)) {
                    module = module1;
                }
                else {
                    module = module2;
                }
                break;
            }
            default:
                throw new RuntimeException("Unsupported module type: " + moduleType);
        }

        Set<OWLAxiom> logicalAxioms = ontologyAxiomSet.getAxiomSet(module);
        Set<OWLEntity> moduleSignature = new HashSet<OWLEntity>();
        for (int i = 0; i < module.length; i++) {
            if (module[i]){
                moduleSignature.addAll(ontologyAxiomSet.getAxiom(i).getSignature());
            }
        }
        Set<OWLAxiom> nonLogicalAxioms = extractNonLogicalAxioms(logicalAxioms, moduleSignature);
        return new SyntacticLocalityModule(this, logicalAxioms, nonLogicalAxioms, module, sig, moduleSignature); 
    }

    /**
     * Extracts a module from the associated ontology for a given signature and the associated module type, and returns the
     * module as a set of axioms. The module will include annotation and declaration axioms for all entities and
     * axioms in it.
     *
     * @param sig the seed signature (set of entities) for the module
     * @return the module
     */
    public Set<OWLAxiom> extract(Set<OWLEntity> sig) {
        return extractModule(sig).getAxioms();
    }


    /**
     * Extracts a module from the associated ontology for a given signature and the associated module type, and returns the
     * module as an ontology.
     *
     * @param signature the seed signature (set of entities) for the module
     * @param uri       the URI for the module
     * @return the module, having the specified URI
     * @throws OWLOntologyChangeException   if adding axioms to the module fails
     * @throws OWLOntologyCreationException if the module cannot be created
     */
    public OWLOntology extractAsOntology(Set<OWLEntity> signature, URI uri) throws OWLOntologyCreationException, OWLOntologyChangeException {
        return extractModule(signature).asOWLOntology(uri);
	}

}
