/*
SME^2 DFKI GmbH 2007
Refer to license.txt for copyright details. 
*/
package de.dfki.owlsmx.sme2;

import de.dfki.owlsmx.data.MatchedService;
import de.dfki.owlsmx.exceptions.MatchingException;
import de.dfki.sme2.IMatchmakerPlugin;
import static de.dfki.owlsmx.sme2.OWLSM3.FileNameFilter.OWLS;
import de.dfki.owlsmx.SimilarityMatchingEngine;

import java.io.*;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Hashtable;
import java.util.SortedSet;
import java.util.Vector;

/**
 * OWLS-MX 1.1 matchmaker plugin.
 *
 * @version 1.0 20/05/07
 * @author jmisutka
 * @version 1.1 24/07/07
 * @author Patrick Kapahnke
 */
public class OWLSM3 implements IMatchmakerPlugin {

    // Logger
    private static java.util.logging.Logger logger = Logger.getLogger ("OWLSMX");


    //=====================================================
    // Constants
    //=====================================================

    private final static double SYN_THRESHOLD = 0.51;

    //=====================================================
    // Variables
    //=====================================================

    private final SimilarityMatchingEngine _mm_impl = new SimilarityMatchingEngine ();

    //
    // optimalization - create it during initialization
    private Hashtable<URL,Vector<URL>> _result = new Hashtable<URL,Vector<URL>>();


    //=====================================================
    // Ctor
    //=====================================================

    /**
     * Ctor.<p>
     *
     * Sets default parameters to OWLSMX matchmaker.
     */
    public OWLSM3 () {
    	_mm_impl.setSimilarityMeasure(de.dfki.owlsmx.similaritymeasures.SimilarityMeasure.SIMILARITY_COSINE);
    }

    //=====================================================
    // IMatchmakerPlugin interface implementation
    //=====================================================

    //
    // input
    //
    public void input (URL serviceURL) {

        try {
            // get service names
            String[]_service_names = names (serviceURL);

            // loop through service names and give them to matchmaker
            for (String service_name : _service_names) {
                try {
                    URI uri = new URI (service_name);

                    // add service
                    _mm_impl.addService(uri);
//                        logger.severe ("Added successfully - "+service_name.substring(service_name.lastIndexOf('/')));

                }catch(Exception e){
                    logger.severe ("Error adding service: " + service_name + " "+e.toString());
                }
            } // for

        } catch (Exception e) {
            logger.severe ("Can't parse services - "+e.toString());
        }
    }

    //
    // query
    //
    public Hashtable<URL,Vector<URL>> query (URL queryURL) {
        // clear before each call
        _result.clear ();

        try {
            // get names
            String[] query_names = names (queryURL);
            //
            // loop through queries and ask the matchmaker for results
            //
            for (int i = 0; i < query_names.length; ++i) {
                    logger.severe ("query");
                Vector<URL> replies = new Vector<URL> ();

                try {
                    URL query = new URL (query_names[i]);
                    SortedSet<MatchedService> reply = _mm_impl.matchRequest (inputstream(query), de.dfki.owlsmx.data.DegreeOfMatch.FAIL, SYN_THRESHOLD);
                    reply = SimilarityMatchingEngine.sortMatchingResult(reply, de.dfki.owlsmx.data.SortingType.HYBRID);

                    // accept / reject reply
                    for (MatchedService result : reply) {
                    	replies.add (result.serviceURI.toURL());
                    }

                    _result.put (query, replies);
                        logger.severe ("finished query");

                }catch (FileNotFoundException e) {
                    System.out.println ("File not found: " + query_names[i]);
                    logger.severe ("error"+e.toString());
                }catch (MatchingException e) {
                    System.out.println ("Can't query: " + query_names[i]);
                    logger.severe ("error"+e.toString());
                } catch (Exception e) {
                    System.out.println ("Invalid url: " + query_names[i]);
                    logger.severe ("error"+e.toString());
                }
            }

        } catch (IOException e) {
            logger.severe ("Can't get queries - "+e.toString());
            return null;
        }

        return _result;
    }



    //=====================================================
    // Helper functions
    //=====================================================

    // local file prefix
    private static final String FILE = "file://";

    //
    // if url is a directory then a container of full url names of files in
    // that directory is returned, otherwise only specified url is returned
    // NOTE: directories can be only local - interface documentation
    //
    private static String[] names (URL url) throws IOException {

        // local file
        if (url.toString().startsWith(FILE)) {

            File file = new File (url.toString().substring (FILE.length()));
            if (file.isDirectory ()) {
                String [] list = file.list (OWLS);
                for (int i=0; i < list.length; ++i)
                    list[i] = url.toString().replaceAll("file://", "file:///") + list[i];
                return list;
            }
        }

        return new String [] {url.toString().replaceAll("file://", "file:///")};
    }


    //
    // returns input stream
    //
    private static InputStream inputstream (URI uri)
                throws IOException {
        // local file
        if (uri.toString().startsWith(FILE))
            return new FileInputStream (uri.toString().substring (FILE.length()));
        else
            return uri.toURL().openStream ();
    }

    //
    // converts URL to URI and calls inputstream (URI)
    //
    private static InputStream inputstream (URL url)
                throws IOException, URISyntaxException {
        return inputstream (url.toURI());
    }


    //=====================================================
    // Helper classes
    //=====================================================

    //
    // file name filter class
    //
    public static class FileNameFilter implements java.io.FilenameFilter {
        // Common used
        public final static FileNameFilter OWLS = new FileNameFilter (".owls");
        // Variables
        private final String _suffix;
        // Ctor
        public FileNameFilter (String suffix) {
            _suffix = suffix;
        }
        // Interface implementation
        public boolean accept (File dir, String name) {
            return name.endsWith (_suffix);
        }

    } // class FileNameFilter

} // class OWLSMX